#include <stdio.h>
#include <errno.h>
#include <unistd.h> 
#include <bpf/libbpf.h> 
#include "load_bpf.h"
#include <string.h> 

struct bpf_object *load_bpf_program_from_file(const char* path)
{
    struct bpf_object *obj;
    if (!path) 
    {
        fprintf(stderr, 
            "\e[0;31m[-] Path is NULL\e[0m\n");
        return NULL;
    }
    if (access(path, R_OK)) 
    {
        fprintf(stderr, 
            "\e[0;31m[-] BPF object file not accessible: %s\e[0m\n", 
            strerror(errno));
        return NULL;
    }
    obj = bpf_object__open_file(path, NULL);
    if (!obj || libbpf_get_error(obj)) 
    {
        fprintf(stderr,
            "\e[0;31m[-] Failed to open BPF object file\e[0m\n");
        return NULL;
    }
    if (bpf_object__load(obj)) 
    {
        fprintf(stderr, 
            "\e[0;31m[-] Failed to load BPF object: %s\e[0m\n", 
            strerror(errno));
        struct bpf_program *p = NULL;
        size_t log_len = 0;
        fprintf(stderr, 
            "\e[0;35m=== BPF verifier logs (if any) ===\e[0m\n");
        p = bpf_object__next_program(obj, NULL);
        while (p) 
        {
            const char *log = bpf_program__log_buf(p, &log_len);
            if (log && log_len) 
            {
                fprintf(stderr, "--- prog '%s' verifier log (len=%zu) ---\n%s\n",
                        bpf_program__name(p) ? bpf_program__name(p) : "(noname)",
                        log_len, log);
            }
            p = bpf_object__next_program(obj, p);
        }
        fprintf(stderr, 
            "\e[0;35m===================================\e[0m\n");

        bpf_object__close(obj);
        return NULL;
    }

    printf("\e[0;34m[+] Load BPF object Success.\e[0m\n");
    return obj;
}